/****************************************************************************
 *
 * Copyright (c) 1998, Network Associates, Inc. and its affiliated Companies
 *
 ****************************************************************************/

#include <sys/types.h> 
#include <time.h>

#include "cms.h"

#define ALG_UNKNOWN 0
#define ALG_RSA 1
#define ALG_DSA 2

static int GetAlgorithm(PKIOBJECT_ID alg)
{
    if (alg.len == TC_ALG_RSA_MD2_LEN &&
	memcmp(alg.val, TC_ALG_RSA_MD2, TC_ALG_RSA_MD2_LEN) == 0)
	return ALG_RSA;

    else if (alg.len == TC_ALG_RSA_MD5_LEN &&
	memcmp(alg.val, TC_ALG_RSA_MD5, TC_ALG_RSA_MD5_LEN) == 0)
	return ALG_RSA;

    else if (alg.len == TC_ALG_RSA_SHA1_LEN &&
	memcmp(alg.val, TC_ALG_RSA_SHA1, TC_ALG_RSA_SHA1_LEN) == 0)
	return ALG_RSA;

    else if (alg.len == TC_ALG_RSA_LEN &&
	memcmp(alg.val, TC_ALG_RSA, TC_ALG_RSA_LEN) == 0)
	return ALG_RSA;

    else if (alg.len == TC_ALG_DSA_SHA1_LEN &&
	     memcmp(alg.val, TC_ALG_DSA_SHA1, TC_ALG_DSA_SHA1_LEN) == 0)
	return ALG_DSA;

    else if (alg.len == TC_ALG_DSA_LEN &&
	     memcmp(alg.val, TC_ALG_DSA, TC_ALG_DSA_LEN) == 0)
	return ALG_DSA;

    else
	return ALG_UNKNOWN;
    
}

#define EXT_UNKNOWN       0
#define EXT_AUTHKEYID     1
#define EXT_SUBJKEYID     2
#define EXT_KEYUSAGE      3
#define EXT_PRIVKEYUSAGE  4
#define EXT_POLICYMAP     5
#define EXT_SUBALTNAME    6
#define EXT_ISSUERALTNAME 7
#define EXT_SUBJDIRATTR   8
#define EXT_BASICCON      9
#define EXT_NAMECON      10 
#define EXT_POLICYCON    11

static int ExtType(PKIOBJECT_ID oid)
{
    if (oid.len == PKIid_ce_authorityKeyIdentifier_OID_LEN &&
	memcmp(oid.val, PKIid_ce_authorityKeyIdentifier_OID,
	       PKIid_ce_authorityKeyIdentifier_OID_LEN) == 0)
	return EXT_AUTHKEYID;

    else if (oid.len == PKIid_ce_subjectKeyIdentifier_OID_LEN &&
	     memcmp(oid.val, PKIid_ce_subjectKeyIdentifier_OID,
		    PKIid_ce_subjectKeyIdentifier_OID_LEN) == 0)
	return EXT_SUBJKEYID;

    else if (oid.len == PKIid_ce_keyUsage_OID_LEN &&
	     memcmp(oid.val, PKIid_ce_keyUsage_OID,
		    PKIid_ce_keyUsage_OID_LEN) == 0)
	return EXT_KEYUSAGE;

    else if (oid.len == PKIid_ce_issuerAltName_OID_LEN &&
	      memcmp(oid.val, PKIid_ce_issuerAltName_OID,
		     PKIid_ce_issuerAltName_OID_LEN) == 0)
	return EXT_ISSUERALTNAME;

    else if (oid.len == PKIid_ce_subjectAltName_OID_LEN &&
	     memcmp(oid.val, PKIid_ce_subjectAltName_OID,
		    PKIid_ce_subjectAltName_OID_LEN) == 0)
	return EXT_SUBALTNAME;

    else if (oid.len == PKIid_ce_privateKeyUsagePeriod_OID_LEN &&
	     memcmp(oid.val, PKIid_ce_privateKeyUsagePeriod_OID,
		    PKIid_ce_privateKeyUsagePeriod_OID_LEN) == 0)
	return EXT_PRIVKEYUSAGE;

    else if (oid.len == PKIid_ce_policyMappings_OID_LEN &&
	     memcmp(oid.val, PKIid_ce_policyMappings_OID,
		    PKIid_ce_policyMappings_OID_LEN) == 0)
	return EXT_POLICYMAP;

    else if (oid.len == PKIid_ce_subjectDirectoryAttributes_OID_LEN &&
	     memcmp(oid.val, PKIid_ce_subjectDirectoryAttributes_OID,
		    PKIid_ce_subjectDirectoryAttributes_OID_LEN) == 0)
	return EXT_SUBJDIRATTR;

    else if (oid.len == PKIid_ce_basicConstraints_OID_LEN &&
	     memcmp(oid.val, PKIid_ce_basicConstraints_OID,
		    PKIid_ce_basicConstraints_OID_LEN) == 0)
	return EXT_BASICCON;

    else if (oid.len == PKIid_ce_nameConstraints_OID_LEN &&
	     memcmp(oid.val, PKIid_ce_nameConstraints_OID,
		    PKIid_ce_nameConstraints_OID_LEN) == 0)
	return EXT_NAMECON;

    else if (oid.len == PKIid_ce_policyConstraints_OID_LEN &&
	     memcmp(oid.val, PKIid_ce_policyConstraints_OID,
		    PKIid_ce_policyConstraints_OID_LEN) == 0)
	return EXT_POLICYCON;

    else
        return EXT_UNKNOWN;
}

static int CheckDname()
{
    /* if its a DirectoryString components, then its must be
       PrintableString, BMPString, or UTF8String; 4.2.1.4, 4.1.2.6,  */


    return 0;
}

static void CheckExtensions(
			    const TC_TBSCertificate *tbsCert,
			    TC_CertType      certType,
			    int              errorList[],
			    int              *numErrors,
			    TC_CONTEXT       *ctx)
{
  int localNumErrors = *numErrors;
  TC_ExtensionList *exts = tbsCert->extensions;
  int i, j;

  /* only one instance of an extension type in a cert; 4.2 */
  for (i = 0; i < exts->n - 1; i++)
    for (j = i+1; j < exts->n; j++)
      if (exts->elt[i]->extnID.len == exts->elt[j]->extnID.len &&
	  memcmp(exts->elt[i]->extnID.val,
		 exts->elt[j]->extnID.val,
		 exts->elt[j]->extnID.len) == 0)
      {
	errorList[localNumErrors] = TC_E_DuplicateExtensionEntry;
	localNumErrors++;
      }


  for (i = 0; i < exts->n; i++) {

    switch(ExtType(exts->elt[i]->extnID))
    {

	case EXT_AUTHKEYID:
	    /* authority key identifier uses the keyIdentifier field; 4.2.1.1 */
	        /* ext. not supported yet */

	    /* authority key identifier not critical; 4.2.1.1 */
	    if (exts->elt[i]->critical != NULL &&
		PKIGetBoolVal(ctx->certasnctx,
				   exts->elt[i]->critical) != PKIFALSE)
	    {
		errorList[localNumErrors] = TC_E_AuthKeyIDMarkedCritical;
		localNumErrors++;
	    }
	    
	    break;

	case EXT_SUBJKEYID:
	    /* subject key identifier must be present in CA certs; 4.2.1.2 */
	    /* subject key identifier should be present in EE certs; 4.2.1.2 */
                    /* ext. not supported yet */

	    /* subject key identifier not critical; 4.2.1.2 */
	    /* policy mappings not critical; 4.2.1.6 */
	    if (exts->elt[i]->critical != NULL &&
		PKIGetBoolVal(ctx->certasnctx,
				   exts->elt[i]->critical) != PKIFALSE) {
		errorList[localNumErrors] = TC_E_SubjectKeyIDMarkedCritical;
		localNumErrors++;
	    }

	    break;

	case EXT_KEYUSAGE:
	    /* key usage is critical; 4.2.1.3 */
	    if (exts->elt[i]->critical == NULL) {
		errorList[localNumErrors] = TC_E_KeyUsageNotMarkedCritical;
		localNumErrors++;
	    }
	    else if (PKIGetBoolVal(ctx->certasnctx,
				   exts->elt[i]->critical) == PKIFALSE) {
		errorList[localNumErrors] = TC_E_KeyUsageNotMarkedCritical;
		localNumErrors++;
	    }

	    break;

	case EXT_PRIVKEYUSAGE:
	    /* do not use private key usage extension; 4.2.1.4 */
	    errorList[localNumErrors] = TC_E_DontUsePrivateKeyUsage;
	    localNumErrors++;

	    break;

	case EXT_POLICYMAP:
	    /* policy mappings not critical; 4.2.1.6 */
	    if (exts->elt[i]->critical != NULL &&
		PKIGetBoolVal(ctx->certasnctx,
				   exts->elt[i]->critical) != PKIFALSE) {
		errorList[localNumErrors] = TC_E_PolicyMappingMarkedCritical;
		localNumErrors++;
	    }

	    break;

	case EXT_SUBALTNAME:
	    /* if subject field is empty then subjectAltName is critical
	       (if present); 4.2.1.7 */
	    /* ip address in sub/issuer altName is either 4 or
	       16 bytes; 4.2.1.7 */
	    /* as least one entry in sub/issuer altName; 4.2.1.7, 4.2.1.8 */
	    /* names sub/issuer altName are not blank or just
	       whitespace; 4.2.1.7, 4.2.1.8 */

	    break;

	case EXT_ISSUERALTNAME:
	    /* issuerAltName is not critical; 4.2.1.8 */
	    if (exts->elt[i]->critical != NULL &&
		PKIGetBoolVal(ctx->certasnctx,
				   exts->elt[i]->critical) != PKIFALSE) {
		errorList[localNumErrors] = TC_E_IssuerAltNameMarkedCritical;
		localNumErrors++;
	    }

	    /* ip address in sub/issuer altName is either 4 or
	       16 bytes; 4.2.1.7 */
	    /* as least one entry in sub/issuer altName; 4.2.1.7, 4.2.1.8 */
	    /* names sub/issuer altName are not blank or just
	       whitespace; 4.2.1.7, 4.2.1.8 */

	    break;
    
	case EXT_SUBJDIRATTR:
	    /* subject directory attributes is not critical; 4.2.1.9 */
	    if (exts->elt[i]->critical != NULL &&
		PKIGetBoolVal(ctx->certasnctx,
				   exts->elt[i]->critical) != PKIFALSE) {
		errorList[localNumErrors] = TC_E_SubjectDirAttrMarkedCritical;
		localNumErrors++;
	    }

	    break;

	case EXT_BASICCON:
	    /* CA certs must have basic constraints; 4.2.1.10 */

	    /* EE certs do not have basic constraints; 4.2.1.10 */
	    if (certType == TC_EndEntity){
		errorList[localNumErrors] = TC_E_EECertWithBasicConstraints;
		localNumErrors++;
	    }

	    /* by default extensions should be false; 4.2 */
	    if (exts->elt[i]->critical != NULL &&
		PKIGetBoolVal(ctx->certasnctx,
				   exts->elt[i]->critical) != PKIFALSE) {
		errorList[localNumErrors] = TC_E_BasicConstraintsMarkedCritical;
		localNumErrors++;
	    }

	    break;

	case EXT_NAMECON:  /* ext. not supported yet */
	    /* name constraints only in CA certs; 4.2.1.11 */
	    if (certType != TC_RootCertAuthority ||
		certType != TC_CertificateAuthority) {
		errorList[localNumErrors] = TC_E_NonCACertWithNameConstraints;
		localNumErrors++;
	    }

	    /* name constraints is critical; 4.2.1.11 */
	    if (exts->elt[i]->critical == NULL) {
		errorList[localNumErrors] = 
		          TC_E_NameConstraintsNotMarkedCritical;
		localNumErrors++;
	    }
	    else if (PKIGetBoolVal(ctx->certasnctx,
				   exts->elt[i]->critical) == PKIFALSE) {
		errorList[localNumErrors] =
		          TC_E_NameConstraintsNotMarkedCritical;
		localNumErrors++;
	    }

	    break;

	case EXT_POLICYCON:
	    /* one field present in policy constraints; 4.2.1.12*/
	        /* ext. not supported yet */

	    /* by default extensions should be false; 4.2 */
	    if (exts->elt[i]->critical != NULL &&
		PKIGetBoolVal(ctx->certasnctx,
				   exts->elt[i]->critical) != PKIFALSE) {
		errorList[localNumErrors] = TC_E_PolicyConstraintsMarkedCritical;
		localNumErrors++;
	    }

	    break;

	default:
	    /* by default extensions should be false; 4.2 */
	    if (exts->elt[i]->critical != NULL &&
		PKIGetBoolVal(ctx->certasnctx,
				   exts->elt[i]->critical) != PKIFALSE) {
		errorList[localNumErrors] = TC_E_UnknownExtensionMarkedCritical;
		localNumErrors++;
	    }

	    break;
	} /* switch */

    }/* for each extension */

    *numErrors = localNumErrors;
    return;
} 

/****
 *
 * tc_CertCheckPKIXCompliance
 *
 * Check the provided certificate for PKIX part 1 compliance.  The paragraph
 * numbers from the spec. where the requirements are located are included
 * in the comments.  This routine checks what it can, but it can't check
 * everything since it only has the certificate information in isolation.  For
 * instance, it can't check that the serial number is unique for the issuing
 * CA.
 *
 * The routine will return the first error found.  It goes through all the
 * checks it can and if you would like all the possible errors, you can provide
 * pointers for numErrors and errorList to receive an array of all the errors
 * found.  You are responsible for deallocating the memory in errorList in
 * this case.
 *
 * If there is a problem in the processing itself (eg., memory allocation
 * problem, then the routine will set  errorList and numErrors to NULL
 * and return an error appropriate to the problem (eg., TC_E_NOMEMORY).
 *
 * Note: this call does not assume the cert was created with tc_create_cert()
 * since the caller could have changed some values on their own.  It does
 * assume that a newly created certificate structure is being checked so
 * it will return errors for items related to newly created certs (eg.,
 * issuer unique identifier should not be present )
 *
 *****/

int tc_CertCheckPKIXCompliance(
    const TC_CERT    *cert,
    TC_CertType      certType,
    int              **errorList,
    int              *numErrors,
    TC_CONTEXT       *ctx)
{
    int localErrors[300];
    int localNumErrors;

    PKIAlgorithmIdentifier  *sigAlg;
    TC_TBSCertificate *tbsCert;
    PKIRDNSequence *rdnSeq;
    PKIAlgorithmIdentifier *keyAlg;
    int ver;
    int error = 0;
    int i;
    
    if (cert == NULL || ctx == NULL)
	return TC_E_INVARGS;

    if (errorList != NULL)
	*errorList = NULL;
    if (numErrors != NULL)
	*numErrors = 0;
    localErrors[0] = 0;
    localNumErrors = 0;

    tbsCert = cert->tbsCertificate;

    /* sigAlg field same as signature field in tbsCert; 4.1.1.2 & 4.1.2.3 */
    sigAlg = &cert->cert->signatureAlgorithm;
    if (tbsCert->signature.algorithm.len != sigAlg->algorithm.len ||
	memcmp(tbsCert->signature.algorithm.val,
	       sigAlg->algorithm.val, sigAlg->algorithm.len) != 0)
    {
	localErrors[localNumErrors] = TC_E_SigAlgSignatureMismatch;
	localNumErrors++;
    }

    /* version value omitted when its v1, version value is v2 when 
       UniqueIdentifier is present, version is v3 when extensions
       present; 4.1.2.1 */
    if (tbsCert->version != NULL)
    {
	ver = PKIGetIntVal(ctx->certasnctx, tbsCert->version, &error);
	if (tbsCert->extensions != NULL &&
	    ver != TC_X509_VERSION_3)
	{
	    localErrors[localNumErrors] = TC_E_VersionShouldBe3;
	    localNumErrors++;
	}
	else if ( (tbsCert->issuerUniqueID != NULL ||
		   tbsCert->subjectUniqueID != NULL) &&
		  ver != TC_X509_VERSION_2)
	{
	    localErrors[localNumErrors]  = TC_E_VersionShouldBe2;
	    localNumErrors++;
	}
	else
	{
	    localErrors[localNumErrors] = TC_E_VersionShouldBeNULL;
	    localNumErrors++;
	}
    }
    else /* NULL version, so its v1 */
    {
	if (tbsCert->extensions != NULL)
	{
	    localErrors[localNumErrors] = TC_E_VersionShouldBe3;
	    localNumErrors++;
	}
	else if (tbsCert->issuerUniqueID != NULL ||
		 tbsCert->subjectUniqueID != NULL )
	{
	    localErrors[localNumErrors] = TC_E_VersionShouldBe2;
	    localNumErrors++;
	}
    }

    /* issuer field is not empty, this assumes the Name CHOICE is
       the RDNSequence type; 4.1.2.4 */
    rdnSeq = (PKIRDNSequence *)tbsCert->issuer.data;
    if (rdnSeq->n <= 0)
    {
	localErrors[localNumErrors] = TC_E_EmptyIssuerName;
	localNumErrors++;
    }

    /* Dname checks */
    /* call CheckDname() with subject */

    /* validity is UTCTime until 2049, assuming current size of time_t
       is 4 bytes, we can reliably check for date until 2038, so fail
       then with different error.  Hopefully this will cause a look
       at the code so it can be updated as needed. See comments in
       tc_encode_utctime() for more details like where the numbers
       below came from. 4.1.2.5 */
    if (time(NULL) > (2147483647L - (31536000/2)) )
    {
	localErrors[localNumErrors] = TC_E_CantHandleCurrentTimeValue;
	localNumErrors++;
    }
    else
    {
	if (tbsCert->validity.notBefore.CHOICE_field_type != 
	                                               PKIID_UTCTime ||
	    tbsCert->validity.notAfter.CHOICE_field_type != 
                                                       PKIID_UTCTime )
	{
	    localErrors[localNumErrors] = TC_E_ValidityNotUTCTime;
	    localNumErrors++;
	}
    }    

    /* CA cert has non-empty subject field; 4.1.2.6 */
    if (certType == TC_RootCertAuthority ||
	certType == TC_CertificateAuthority)
    {

	rdnSeq = (PKIRDNSequence *)tbsCert->subject.data;
	if (rdnSeq->n <= 0)
	{
	    localErrors[localNumErrors] = TC_E_EmptyIssuerName;
	    localNumErrors++;
	}
    }

    /* root cert has same issuer/subject fields; 4.1.2.6 */
    if (certType == TC_RootCertAuthority)
    {
	if (tc_compare_dname(&tbsCert->issuer, &tbsCert->subject,
			     ctx) != 1)
	{
	    localErrors[localNumErrors] = TC_E_IssuerSubjectNotSameInRootCert;
	    localNumErrors++;
	}
    }

    /* new certs should not have subject/issuer unique identifier; 4.1.2.7 */
    if (tbsCert->issuerUniqueID != NULL)
    {
	localErrors[localNumErrors] = TC_E_IssuerUniqueIDPresent;
	localNumErrors++;
    }
    if (tbsCert->subjectUniqueID != NULL)
    {
	localErrors[localNumErrors] = TC_E_SubjectUniqueIDPresent;
	localNumErrors++;
    }

    /* Extension checks */
    CheckExtensions(tbsCert, certType, localErrors, &localNumErrors, ctx);

    /* if its RSA, parameters must be present and NULL in the
       AlgorithmIdentifier field, need to check each field; 7.2.1 */

    /* the signature field in TBSCertificate */
    sigAlg = &tbsCert->signature;
    if ( GetAlgorithm(sigAlg->algorithm) == ALG_RSA)
    {
	if (sigAlg->parameters == NULL)
	{
	    localErrors[localNumErrors] = TC_E_RSAParametersMissing;
	    localNumErrors++;
	}
	else if (sigAlg->parameters->len < 2 ||
		 sigAlg->parameters->val[0] != 0x50)
	{
	    localErrors[localNumErrors] = TC_E_RSAParametersNotASN1NULL;
	    localNumErrors++;
	}
    }

    /* the subjectPublicKeyInfo field in TBSCertificate */
    keyAlg = &tbsCert->subjectPublicKeyInfo.algorithm;
    if ( GetAlgorithm(keyAlg->algorithm) == ALG_RSA)
    {
	if (keyAlg->parameters == NULL)
	{
	    localErrors[localNumErrors] = TC_E_RSAParametersMissing;
	    localNumErrors++;
	}
	else if (keyAlg->parameters->len < 2 ||
		 keyAlg->parameters->val[0] != 0x50)
	{
	    localErrors[localNumErrors] = TC_E_RSAParametersNotASN1NULL;
	    localNumErrors++;
	}
    }

    /* the signatureAlgorithm field in Certificate */
    sigAlg = &cert->cert->signatureAlgorithm;
    if ( GetAlgorithm(sigAlg->algorithm) == ALG_RSA)
    {
	if (sigAlg->parameters == NULL)
	{
	    localErrors[localNumErrors] = TC_E_RSAParametersMissing;
	    localNumErrors++;
	}
	else if (sigAlg->parameters->len < 2 ||
		 sigAlg->parameters->val[0] != 0x50)
	{
	    localErrors[localNumErrors] = TC_E_RSAParametersNotASN1NULL;
	    localNumErrors++;
	}
    }

    /* if its DSA, parameters field absent in AlgorithmIdentifier field
       for the signature field in TBSCertificate and the signatureAlgorithm
       field in Certificate; 7.2.2 */
    sigAlg = &tbsCert->signature;
    if ( GetAlgorithm(sigAlg->algorithm) == ALG_DSA)
    {
	if (sigAlg->parameters != NULL)
	{
	    localErrors[localNumErrors] = TC_E_DSAParametersArePresent;
	    localNumErrors++;
	}
    }

    sigAlg = &cert->cert->signatureAlgorithm;
    if ( GetAlgorithm(sigAlg->algorithm) == ALG_DSA)
    {
	if (sigAlg->parameters != NULL)
	{
	    localErrors[localNumErrors] = TC_E_DSAParametersArePresent;
	    localNumErrors++;
	}
    }

    /* create the error list to return if requested */
    if (errorList != NULL && numErrors != NULL)
    {
	*errorList = (int *)TC_Alloc(ctx->memMgr, sizeof(int) * localNumErrors);
	if (errorList == NULL)
	    return TC_E_NOMEMORY;

	for (i = 0; i < localNumErrors; i++)
	    (*errorList)[i] = localErrors[i];
	*numErrors = localNumErrors;
    }

    return localErrors[0]; /* return the first error found */

} /* tc_CertCheckPKIXCompliance */


/****
 *
 * tc_CRLCheckPKIXCompliance
 *
 *****/
int tc_CRLCheckPKIXCompliance(
    const TC_CertificateList *crl,
    TC_CRLType  crlType,
    TC_CONTEXT *ctx)
{
    int status = 0;

    if (crl == NULL || ctx == NULL)
	return TC_E_INVARGS;

    /* version is v2 */

    /* issuer field is not empty, this assumes the Name CHOICE is
       the RDNSequence type; 5.1.3.2 */

    /* include nextUpdate field */

    /* include CRL number extension */

    /* include authority key identifier field */

    /* signatureAlgorithm field same as signature field in tbsCertList */

    /* issuer name is present */

    /* thisUpdate and nextUpdate are UTCTime until 2049 */

    /* nextUpdate later than thisUpdate */

    /* issuing distribution point ext is critical if there */

    /* reason code ext is not critical */

    /* don't use 'unspecified' in reason code ext. */

    /* hold instruction code is not critical */

    /* hold instruction codes recognized are (see para 5.3.2) */

    /* invalidity data ext is not critical */

    /* cert issuer CRL entry ext. is critical */

    /* if its RSA, parameters must be present and NULL */

    /* if its DSA, parameters field abscent in AlgorithmIdentifier field */

    return status;

} /* tc_CRLCheckPKIXCompliance */
